/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
using System;
using System.Threading;
using Inet.Viewer;
using Inet.Viewer.Resources;

namespace Inet.Viewer.Data
{

    /// <summary>
    /// Implements the loading of the GroupTree and their nodes
    /// </summary>
    public class GroupLoader : Loader
    {
        private const string EmptyGroupName = "    ";
        /// <summary>
        /// 
        /// </summary>
        public const string UrlParameterSubreport = "subreport";
        /// <summary>
        /// 
        /// </summary>
        public const string UrlParameterTypeAndInstance = "type=3&instance=";

        private GroupTreeNode[] groupTreeNodes;

        /// <summary>
        /// The array with GroupTreeNodes, loaded by this class
        /// </summary>
        public GroupTreeNode[] GroupTreeNodes
        {
            get
            {
                return groupTreeNodes;
            }
        }

        /// <summary>
        /// Starts the Thread for parsing the groupTree data </summary>  
        public virtual void LoadData()
        {
            Offset = 0;
            groupTreeNodes = null;
            try
            {
                if (Data != null)
                {
                    ReadTokens();
                }
            }
            catch (Exception ex)
            {
                throw ViewerException.CreateViewerExceptionWithMessage(strings.ErrorMessage_Grouptree_DataCouldNotBeRead, ex);
            }
        }

        /// <summary>
        /// Reading for the whole token TOKEN_GROUPTREE including all the sub nodes        
        /// <param name="maintoken"> ID of the token to be read</param>
        /// <returns>To indicate if the loading is to be continued. True: continue, false: stop loading </returns>
        /// </summary>
        /// <seealso cref= "Inet.Viewer.Data.Loader.Load(int)"/>
        protected internal override bool Load(int maintoken)
        {
            if (maintoken == ViewerTokenConstants.TokenGroupTree)
            {
                int nodeNumber = ReadInt();
                groupTreeNodes = new GroupTreeNode[nodeNumber];
                for (int i = 0; i < nodeNumber && Offset < Data.Length; i++)
                {
                    int token = ReadInt();
                    if (token != ViewerTokenConstants.TokenGroupTreeNode)
                    {
                        throw new ViewerException(strings.ErrorMessage_Grouptree_NodeExpected);
                    }
                    TokenSize = ReadInt();
                    OldOffset = Offset;
                    ReadNode(i);
                    Offset = OldOffset + TokenSize;
                }
            }
            else
            {
                return base.Load(maintoken); // case 1: Error message, usw.
            }
            return true;
        }

        /// <summary>
        /// Parses the data of one node and adds it to the internal nodesArray</summary>
        /// <param name="index"> Offset </param>
        private void ReadNode(int index)
        {
            byte level = (byte)ReadInt();
            byte requesttype = (byte)ReadInt();
            int groupNumber = ReadInt();
            string requestURL = ReadString();
            string groupName = ReadString();

            if (groupName.Length == 0)
            {
                groupName = EmptyGroupName;
            }

            int yFrom = ReadInt();
            int yTo = ReadInt();
            int pageFrom = ReadInt();
            int pageTo = ReadInt();

            GroupTreeNode node = new GroupTreeNode();
            node.SetName(level, requesttype, groupNumber, requestURL, groupName);
            node.SetPosition(yFrom, yTo, pageFrom, pageTo);
            groupTreeNodes[index] = node;
        }

        /// <summary>
        /// Creates a copy of the ReportData and sets the subreport Parameter to show the drill-down report </summary>
        /// <param name="node"> DrillDown-Knoten </param>
        /// <param name="renderData"> </param>
        /// <returns> instance of RendererData for reading of data </returns>
        /// <seealso cref= "Inet.Viewer.IRenderData"/>
        public static IRenderData GetDrillDownRenderDataCopy(GroupTreeNode node, IRenderData renderData)
        {
            IRenderData copy = (IRenderData)renderData.Clone();
            string parameter = UrlParameterTypeAndInstance + node.GroupNumber + node.RequestURL;
            copy[UrlParameterSubreport] = parameter;
            copy.ReportTitle = node.GroupName;
            return copy;
        }
    }
}